import processing.data.JSONObject;
import javax.sound.midi.*;
import java.io.File;
import java.io.FilenameFilter;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import processing.core.PImage;
import java.io.PrintWriter;
import java.util.Scanner;


// Define the time signature and bars
int bpm = 170;
int inputBars = 16; //only important if it can't be read out of your MIDI file

// Set paths for files and folders
String   midiFileName = "LIGHT_BREAK_OB57_8Bar";
String midiFilePath;
String customGraphicsFolderPath;
String exportFolderPath;
String settingsFilePath;

// Adjust the canvas dimensions
float canvasWidthMM = 512;
float canvasHeightMM = 64;

// Adjust the canvas DPI
int dpi = 300; // Set to 300 for final export
//int dpi = 200; // Use 200 for preview purposes

//Change the Processing renderer
// Options: "default", "JAVA2D", "P2D", "P3D"
// Change to "P2D" if you want to use P2D
String renderer = "default"; 


// Schrift Einstellungen
String fontName = "Arial Unicode MS";
int fontSize = 32;
int buttonFontSize = 15;

// Button und Slider Größen und Positionierung
float buttonWidth = 100; // Feste Breite für Buttons
float buttonHeight = 40; // Feste Höhe für Buttons
float buttonSpacing = 10; // Abstand zwischen Buttons
float sliderSpacing = 40; // Erhöhter Abstand zwischen Slidern
float sidePadding = 10; // Verringerter Abstand zum Rand
int radiusButton = 400;
float offsetLabel = 10;
float strokeSize = 0.5;
float arrowStroke = 1.5;
int rotationAngle = 0; // Standardwert


int dynamicButtonRadiusMax = 10;

int bars;
int canvasWidth;
int canvasHeight;
boolean saveButtonHovered = false;
boolean toggleGridButtonHovered = false;
boolean toggleNoteButtonHovered = false;
boolean toggleUnicodeButtonHovered = false;
boolean toggleGraphicsButtonHovered = false;
boolean toggleUIButtonHovered = false;
boolean toggleNameButtonHovered = false;
boolean resetValuesButtonHovered = false;
boolean randomizeButtonHovered = false;
boolean showGrid = false;
boolean showNotes = true;
boolean unicodeMode = false;
boolean circleMode = false;
boolean graphicsMode = true;
boolean showUI = true;
boolean showName = true;
boolean notificationVisible = false;
int notificationStartTime = 0;
Sequence sequence;
int ppq; // Pulses per quarter note
PFont unicodeFont;
PFont buttonFont;
char[] unicodeSymbols = { '⃝', '⃞', '⃟', '⁀', '‿', '↦', '↬', '⊗', '﴿', '﴾', '►', '░', '▓', '〤', '↓', '◄', '₽', '₼', '₰', '▓', '▇', 'ﻷ', 'שּׂ', 'ךּ', 'ᆕ', '৩', 'ﻼ', '₩', '―' };

// Standardwerte
int defaultSize = 100;
int defaultOffsetX = 10;
int defaultOffsetY = 0;
float defaultYSpacing = 1.0f;
int defaultVariance = 10;
float defaultVelocityInfluence = 1.0f;
float defaultRotationStrength = 0.0f;
float defaultRandomizeY = 0.0f;

// Slider Min- und Max-Werte
int minSize = 1, maxSize = 800;
int minOffsetX = -200, maxOffsetX = 1000;
int minOffsetY = -5000, maxOffsetY = 1000;
float minYSpacing = 0.1f, maxYSpacing = 10.0f;
int minVariance = 0, maxVariance = 400;
float minVelocityInfluence = 0.0f, maxVelocityInfluence = 2.0f;
float minRotationStrength = 0.0f, maxRotationStrength = 1.0f;
float minRandomizeY = 0.0f, maxRandomizeY = 400.0f;

// Skalierungsfaktoren
float multiCircle = 0.3;
float multiGraphic = 4.0;
float multiUnicode = 0.8;

// Aktuelle Werte
int size = defaultSize;
int offsetX = defaultOffsetX;
int offsetY = defaultOffsetY;
float ySpacing = defaultYSpacing;
int variance = defaultVariance;
float velocityInfluence = defaultVelocityInfluence;
float rotationStrength = defaultRotationStrength;
float randomizeY = defaultRandomizeY;

Slider sizeSlider; // Slider für Größe
Slider velocityInfluenceSlider; // Slider für Velocity Einfluss
Slider varianceSlider; // Slider für Varianz
Slider offsetXSlider; // Slider für X-Achsen Verschiebung
Slider offsetYSlider; // Slider für Y-Achsen Verschiebung
Slider ySpacingSlider; // Slider für Y-Achsen Abstände
Slider rotationStrengthSlider; // Slider für Rotationsstärke
Slider randomizeYSlider; // Slider für Y-Randomisierung
Slider rotationSlider; // Neuer Slider für Rotation


PImage[] graphicsImages;
int graphicsCount;

// Karte zur Speicherung der Rotationen für jede MIDI Note
Map<String, Float> noteRotations = new HashMap<>();
Map<String, Float> noteYOffsets = new HashMap<>(); // Neue Karte zur Speicherung der Y-Offsets für jede Note

void settings() {

    midiFilePath = sketchPath("MIDI/" + midiFileName + ".mid");
    bars = getBarsFromFileName(midiFilePath, inputBars); // inputBars als Standardwert verwenden


    // Setze relative Pfade für Export und Settings

    exportFolderPath = sketchPath("Export");
    settingsFilePath = sketchPath("settings.txt");
    //midiFilePath = sketchPath("MIDI");
    customGraphicsFolderPath = sketchPath("Graphics");
    
    println("Export Folder Path: " + exportFolderPath);
    println("Settings File Path: " + settingsFilePath);
    println("MIDI File Path: " + midiFilePath);

    
    
    // Berechne die Größe der Leinwand in Pixel basierend auf dem DPI und der Anzahl der Takte
    canvasWidth = (int) (bars * canvasWidthMM / 16.0 * dpi / 25.4); // 512mm für 16 Takte, umgerechnet in Pixel
    canvasHeight = (int) (canvasHeightMM * dpi / 25.4); // 64mm, umgerechnet in Pixel

    // Setze den Renderer basierend auf der definierten Variable
    if (renderer.equals("P2D")) {
        size(canvasWidth, canvasHeight, P2D);
    } else {
        size(canvasWidth, canvasHeight);
    }
}


void setup() {
    println("Leinwandgröße in Pixel: " + canvasWidth + "x" + canvasHeight);
    println("Lade MIDI Datei von: " + midiFilePath);

    // Lade die Schriftart für Unicode Zeichen
    unicodeFont = createFont(fontName, fontSize);
    buttonFont = createFont(fontName, buttonFontSize);

    // Lade die Grafiken aus dem benutzerdefinierten Grafikordner
    loadGraphics();

    // Lade die gespeicherten Einstellungen
    loadSettings();

    // Lade die MIDI Datei und berechne deren Länge
    try {
        File midiFile = new File(midiFilePath);
        if (midiFile.exists()) {
            // Zeige Dateiinformationen an
            println("MIDI-Auto-Read:");
            sequence = MidiSystem.getSequence(midiFile);
            ppq = sequence.getResolution();
            long tickLength = sequence.getTickLength();
            println("PPQ (Pulses per Quarter Note): " + ppq);
            println("Tick Length: " + tickLength);

            // Zeige Benutzereingaben an
            int beatsPerBar = 4; // 4 Schläge pro Takt
            println("\nBenutzereingabe:");
            println("BPM: " + bpm);
            println("Takte: " + bars);
            println("Gesamtschläge (Benutzereingabe): " + (bars * beatsPerBar));

            // Erstelle die Slider
            float sliderWidth = canvasWidth - buttonWidth - sidePadding - sidePadding - dynamicButtonRadiusMax - buttonSpacing; // Passe die Breite des Sliders an, um zwischen Buttons und linker Kante zu passen
            float sliderHeight = 20;
            sizeSlider = new Slider(sidePadding, height - sliderSpacing * 9, sliderWidth, sliderHeight, minSize, maxSize, size);
            velocityInfluenceSlider = new Slider(sidePadding, height - sliderSpacing * 8, sliderWidth,sliderHeight, minVelocityInfluence, maxVelocityInfluence, velocityInfluence);
            varianceSlider = new Slider(sidePadding, height - sliderSpacing * 7, sliderWidth, sliderHeight, minVariance, maxVariance, variance);
            offsetXSlider = new Slider(sidePadding, height - sliderSpacing * 6, sliderWidth, sliderHeight, minOffsetX, maxOffsetX, offsetX);
            offsetYSlider = new Slider(sidePadding, height - sliderSpacing * 5, sliderWidth, sliderHeight, minOffsetY, maxOffsetY, offsetY);
            ySpacingSlider = new Slider(sidePadding, height - sliderSpacing * 4, sliderWidth, sliderHeight, minYSpacing, maxYSpacing, ySpacing);
            rotationSlider = new Slider(sidePadding, height - sliderSpacing * 3, sliderWidth, sliderHeight, 0, 360, rotationAngle);
            rotationStrengthSlider = new Slider(sidePadding, height - sliderSpacing * 2, sliderWidth, sliderHeight, minRotationStrength, maxRotationStrength, rotationStrength);
            randomizeYSlider = new Slider(sidePadding, height - sliderSpacing, sliderWidth, sliderHeight, minRandomizeY, maxRandomizeY, randomizeY); // Hinzugefügt: Slider für Y-Randomisierung

            randomizeNoteAttributes(); // Initiale Randomisierung der Notenattribute
        } else {
            println("MIDI Datei nicht gefunden unter: " + midiFilePath);
        }
    } catch (Exception e) {
        println("Fehler beim Laden der MIDI Datei: " + e.getMessage());
        e.printStackTrace();
    }
}

void draw() {
    background(255); // Weißer Hintergrund


    if (showGrid) {
        drawGrid();
    }

    if (showNotes) {
        if (circleMode) {
            drawCircleNotes();
        } else if (unicodeMode) {
            drawUnicodeNotes();
        } else if (graphicsMode) {
            drawGraphicsNotes();
        }
    }

    if (showName) {
        drawFileName();
    }

    if (showUI) {
        // Zeichne die Buttons unten rechts
        drawButtons();

        // Zeige die Slider an
        sizeSlider.display();
        velocityInfluenceSlider.display();
        varianceSlider.display();
        offsetXSlider.display();
        offsetYSlider.display();
        ySpacingSlider.display();
        rotationSlider.display();

        rotationStrengthSlider.display();
        randomizeYSlider.display(); // Hinzugefügt: Y-Randomisierungs-Slider anzeigen
    }

    if (notificationVisible) {
        drawNotification();
    }
}

void drawGrid() {
    int beatsPerBar = 4;
    float barWidth = width / (float) bars; // Breite eines Takts in Pixel
    float beatWidth = barWidth / beatsPerBar; // Breite eines Schlags in Pixel

    stroke(0); // Setze Strichfarbe auf Schwarz
    strokeWeight(1);
    for (int i = 0; i <= bars * beatsPerBar; i++) {
        float x = i * beatWidth;
        if (i % beatsPerBar == 0) {
            strokeWeight(3); // Dicke Linien für Takte
        } else {
            strokeWeight(1); // Dünne Linien für Schläge
        }
        line(x, 0, x, height);
    }
}


void drawFileName() {
    fill(0);
    textFont(buttonFont);
    textAlign(LEFT, TOP);
    String fileName = new File(midiFilePath).getName();
    text(fileName, 10, 10);
}


// Beispiel für drawCircleNotes mit Rotation
void drawCircleNotes() {
    int beatsPerBar = 4;
    float barWidth = width / (float) bars;
    float beatWidth = barWidth / beatsPerBar;
    float rotationAngle = rotationSlider.value; // Rotationswert vom Slider


    for (Track track : sequence.getTracks()) {
        for (int i = 0; i < track.size(); i++) {
            MidiEvent event = track.get(i);
            MidiMessage message = event.getMessage();
            if (message instanceof ShortMessage) {
                ShortMessage sm = (ShortMessage) message;
                if (sm.getCommand() == ShortMessage.NOTE_ON) {
                    int key = sm.getData1();
                    int velocity = sm.getData2();
                    if (velocity > 0) {
                        long tick = event.getTick();
                        float x = map(tick, 0, bars * 4 * ppq, 0, width);
                        float y = map(key, 0, 127, height, 0) * ySpacing + getNoteYOffset(tick, key); // Wende den skalierte Y-Offset an
                        float adjustedSize = multiCircle * size * (1 + (velocity * velocityInfluence / 127.0));
                        float rotation = getNoteRotation(tick, key) * rotationStrength;
                        pushMatrix();
                        translate(x + offsetX, y + offsetY);
                        rotate(radians(rotationAngle)); // Rotiert die Grafik um den angegebenen Winkel
                        rotate(rotation);
                        fill(0);
                        noStroke();
                        ellipse(0, 0, adjustedSize, adjustedSize);
                        popMatrix();
                    }
                }
            }
        }
    }
}

// Beispiel für drawUnicodeNotes mit Rotation
void drawUnicodeNotes() {
    int beatsPerBar = 4;
    float barWidth = width / (float) bars;
    float beatWidth = barWidth / beatsPerBar;
    float rotationAngle = rotationSlider.value; // Rotationswert vom Slider

    for (Track track : sequence.getTracks()) {
        for (int i = 0; i < track.size(); i++) {
            MidiEvent event = track.get(i);
            MidiMessage message = event.getMessage();
            if (message instanceof ShortMessage) {
                ShortMessage sm = (ShortMessage) message;
                if (sm.getCommand() == ShortMessage.NOTE_ON) {
                    int key = sm.getData1();
                    int velocity = sm.getData2();
                    if (velocity > 0) {
                        long tick = event.getTick();
                        float x = map(tick, 0, bars * 4 * ppq, 0, width);
                        float y = map(key, 0, 127, height, 0) * ySpacing + getNoteYOffset(tick, key); // Wende den skalierte Y-Offset an
                        float adjustedSize = multiUnicode * size * (1 + (velocity * velocityInfluence / 127.0));
                        float rotation = getNoteRotation(tick, key) * rotationStrength;
                        pushMatrix();
                        translate(x + offsetX, y + offsetY);
                        translate(0, -adjustedSize / 2); // Verschiebt den Ursprungspunkt der Translation
                        rotate(radians(rotationAngle)); // Rotiert die Grafik um den angegebenen Winkel
                        rotate(rotation);
                        fill(0);
                        textFont(unicodeFont);
                        textAlign(LEFT, CENTER); // Text wird am linken Rand und zentriert ausgerichtet
                        textSize(adjustedSize);
                        int symbolIndex = (key + variance) % unicodeSymbols.length;
                        text(unicodeSymbols[symbolIndex], 0, 0);
                        popMatrix();
                    }
                }
            }
        }
    }
}

// Beispiel für drawGraphicsNotes mit Rotation
void drawGraphicsNotes() {
    if (graphicsCount == 0) {
        println("Keine Grafiken im CustomGraphics-Ordner gefunden.");
        return;
    }

    int beatsPerBar = 4;
    float barWidth = width / (float) bars;
    float beatWidth = barWidth / beatsPerBar;
    float rotationAngle = rotationSlider.value; // Rotationswert vom Slider

    for (Track track : sequence.getTracks()) {
        for (int i = 0; i < track.size(); i++) {
            MidiEvent event = track.get(i);
            MidiMessage message = event.getMessage();
            if (message instanceof ShortMessage) {
                ShortMessage sm = (ShortMessage) message;
                if (sm.getCommand() == ShortMessage.NOTE_ON) {
                    int key = sm.getData1();
                    int velocity = sm.getData2();
                    if (velocity > 0) {
                        long tick = event.getTick();
                        float x = map(tick, 0, bars * 4 * ppq, 0, width);
                        float y = map(key, 0, 127, height, 0) * ySpacing + getNoteYOffset(tick, key); // Wende den skalierte Y-Offset an
                        float scaleFactor = 1 + (velocity * velocityInfluence / 127.0);
                        float adjustedWidth = multiGraphic * size * scaleFactor;
                        float adjustedHeight = adjustedWidth * ((float) graphicsImages[0].height / graphicsImages[0].width);
                        float rotation = getNoteRotation(tick, key) * rotationStrength;

                        float translateX = x + offsetX;
                        float translateY = y + offsetY;

                        pushMatrix();
                        translate(translateX, translateY);
                        translate(0, -adjustedHeight / 2); // Verschiebt den Ursprungspunkt der Translation
                        rotate(radians(rotationAngle)); // Rotiert die Grafik um den angegebenen Winkel
                        rotate(rotation);
                        
                        imageMode(CORNER);
                        image(graphicsImages[(key + variance) % graphicsCount], 0, 0, adjustedWidth, adjustedHeight);
                        popMatrix();
                    }
                }
            }
        }
    }
}


void drawButtons() {
    float x = canvasWidth - buttonWidth-sidePadding;
    float saveButtonY = canvasHeight - buttonHeight - sidePadding;
    float resetValuesButtonY = saveButtonY - buttonHeight - buttonSpacing;
    float randomizeButtonY = resetValuesButtonY - buttonHeight - buttonSpacing;
    float toggleNoteButtonY = randomizeButtonY - buttonHeight - buttonSpacing;
    float toggleUnicodeButtonY = toggleNoteButtonY - buttonHeight - buttonSpacing;
    float toggleGraphicsButtonY = toggleUnicodeButtonY - buttonHeight - buttonSpacing;
    float toggleNameButtonY = toggleGraphicsButtonY - buttonHeight - buttonSpacing;
    float toggleGridButtonY = toggleNameButtonY - buttonHeight - buttonSpacing;
    float toggleUIButtonY = toggleGridButtonY - buttonHeight - buttonSpacing;

    // Zeichne den Toggle UI Button
    if (mouseX >= x && mouseX <= x + buttonWidth && mouseY >= toggleUIButtonY && mouseY <= toggleUIButtonY + buttonHeight) {
        fill(200);
        toggleUIButtonHovered = true;
    } else {
        fill(255);
        toggleUIButtonHovered = false;
    }

    stroke(0);
    strokeWeight(strokeSize);
    rect(x, toggleUIButtonY, buttonWidth, buttonHeight, radiusButton);

    fill(0);
    textAlign(CENTER, CENTER);
    textFont(buttonFont);
    text("Hide UI", x + buttonWidth / 2, toggleUIButtonY + buttonHeight / 2);

    // Zeichne die anderen Buttons, wenn UI angezeigt wird
    if (showUI) {
        // Zeichne den Toggle Grid Button
        if (mouseX >= x && mouseX <= x + buttonWidth && mouseY >= toggleGridButtonY && mouseY <= toggleGridButtonY + buttonHeight) {
            fill(200);
            toggleGridButtonHovered = true;
        } else {
            fill(255);
            toggleGridButtonHovered = false;
        }

        stroke(0);
        rect(x, toggleGridButtonY, buttonWidth, buttonHeight, radiusButton);

        fill(0);
        textAlign(CENTER, CENTER);
        text("Toggle Grid", x + buttonWidth / 2, toggleGridButtonY + buttonHeight / 2);

        // Zeichne den Toggle Name Button
        if (mouseX >= x && mouseX <= x + buttonWidth && mouseY >= toggleNameButtonY && mouseY <= toggleNameButtonY + buttonHeight) {
            fill(200);
            toggleNameButtonHovered = true;
        } else {
            fill(255);
            toggleNameButtonHovered = false;
        }

        stroke(0);
        rect(x, toggleNameButtonY, buttonWidth, buttonHeight, radiusButton);

        fill(0);
        textAlign(CENTER, CENTER);
        text("Toggle Name", x + buttonWidth / 2, toggleNameButtonY + buttonHeight / 2);

        // Zeichne den Toggle Graphics Button
        if (mouseX >= x && mouseX <= x + buttonWidth && mouseY >= toggleGraphicsButtonY && mouseY <= toggleGraphicsButtonY + buttonHeight) {
            fill(200);
            toggleGraphicsButtonHovered = true;
        } else {
            fill(255);
            toggleGraphicsButtonHovered = false;
        }

        stroke(0);
        rect(x, toggleGraphicsButtonY, buttonWidth, buttonHeight, radiusButton);

        fill(0);
        textAlign(CENTER, CENTER);
        text("Graphics", x + buttonWidth / 2, toggleGraphicsButtonY + buttonHeight / 2);

        // Zeichne den Toggle Unicode Button
        if (mouseX >= x && mouseX <= x + buttonWidth && mouseY >= toggleUnicodeButtonY && mouseY <= toggleUnicodeButtonY + buttonHeight) {
            fill(200);
            toggleUnicodeButtonHovered = true;
        } else {
            fill(255);
            toggleUnicodeButtonHovered = false;
        }

        stroke(0);
        rect(x, toggleUnicodeButtonY, buttonWidth, buttonHeight, radiusButton);

        fill(0);
        textAlign(CENTER, CENTER);
        text("Unicode", x + buttonWidth / 2, toggleUnicodeButtonY + buttonHeight / 2);

        // Zeichne den Toggle Note Button
        if (mouseX >= x && mouseX <= x + buttonWidth && mouseY >= toggleNoteButtonY && mouseY <= toggleNoteButtonY + buttonHeight) {
            fill(200);
            toggleNoteButtonHovered = true;
        } else {
            fill(255);
            toggleNoteButtonHovered = false;
        }

        stroke(0);
        rect(x, toggleNoteButtonY, buttonWidth, buttonHeight, radiusButton);

        fill(0);
        textAlign(CENTER, CENTER);
        text("Circle", x + buttonWidth / 2, toggleNoteButtonY + buttonHeight / 2);

        // Zeichne den Randomize Button
        if (mouseX >= x && mouseX <= x + buttonWidth && mouseY >= randomizeButtonY && mouseY <= randomizeButtonY + buttonHeight) {
            fill(200);
            randomizeButtonHovered = true;
        } else {
            fill(255);
            randomizeButtonHovered = false;
        }

        stroke(0);
        rect(x, randomizeButtonY, buttonWidth, buttonHeight, radiusButton);

        fill(0);
        textAlign(CENTER, CENTER);
        text("Randomize", x + buttonWidth / 2, randomizeButtonY + buttonHeight / 2);

        // Zeichne den Reset Values Button
        if (mouseX >= x && mouseX <= x + buttonWidth && mouseY >= resetValuesButtonY && mouseY <= resetValuesButtonY + buttonHeight) {
            fill(200);
            resetValuesButtonHovered = true;
        } else {
            fill(255);
            resetValuesButtonHovered = false;
        }

        stroke(0);
        rect(x, resetValuesButtonY, buttonWidth, buttonHeight, radiusButton);

        fill(0);
        textAlign(CENTER, CENTER);
        text("Reset Values", x + buttonWidth / 2, resetValuesButtonY + buttonHeight / 2);

        // Zeichne den Save Button
        if (mouseX >= x && mouseX <= x + buttonWidth && mouseY >= saveButtonY && mouseY <= saveButtonY + buttonHeight) {
            fill(200);
            saveButtonHovered = true;
        } else {
            fill(255);
            saveButtonHovered = false;
        }

        stroke(0);
        rect(x, saveButtonY, buttonWidth, buttonHeight, radiusButton);

        fill(0);
        textAlign(CENTER, CENTER);
        text("Save", x + buttonWidth / 2, saveButtonY + buttonHeight / 2);

        // Zeichne die Slider Labels
        drawSliderLabel("Size (" + String.format("%.2f", sizeSlider.value) + ")", sizeSlider.x, sizeSlider.y - offsetLabel);
        drawSliderLabel("Velocity Influence (" + String.format("%.2f", velocityInfluenceSlider.value) + ")", velocityInfluenceSlider.x, velocityInfluenceSlider.y - offsetLabel);
        drawSliderLabel("Variance (" + (int)varianceSlider.value + ")", varianceSlider.x, varianceSlider.y - offsetLabel); // Varianz als int anzeigen
        drawSliderLabel("Offset X (" + (int)offsetXSlider.value + ")", offsetXSlider.x, offsetXSlider.y - offsetLabel); // Offset X als int anzeigen
        drawSliderLabel("Offset Y (" + (int)offsetYSlider.value + ")", offsetYSlider.x, offsetYSlider.y - offsetLabel); // Offset Y als int anzeigen
        drawSliderLabel("Y Spacing (" + String.format("%.2f", ySpacingSlider.value) + ")", ySpacingSlider.x, ySpacingSlider.y - offsetLabel);
        drawSliderLabel("Rotation (" + (int)rotationSlider.value + "°)", rotationSlider.x, rotationSlider.y - offsetLabel);
        drawSliderLabel("Randomize Rotation (" + String.format("%.2f", rotationStrengthSlider.value) + ")", rotationStrengthSlider.x, rotationStrengthSlider.y - offsetLabel);
        drawSliderLabel("Randomize Y (" + String.format("%.2f", randomizeYSlider.value) + ")", randomizeYSlider.x, randomizeYSlider.y - offsetLabel);
    }
}

void drawSliderLabel(String label, float x, float y) {
    textFont(buttonFont);
    textAlign(LEFT, CENTER);
    float w = textWidth(label);
    float h = buttonFontSize * 1.5;
    fill(255);
    rect(x - 5, y - h / 2 + 1, w + 10, h + 2, radiusButton);
    fill(0);
    text(label, x, y);
}

void loadGraphics() {
    File customGraphicsFolder = new File(customGraphicsFolderPath);
    if (customGraphicsFolder.exists() && customGraphicsFolder.isDirectory()) {
        File[] files = customGraphicsFolder.listFiles(new FilenameFilter() {
            public boolean accept(File dir, String name) {
                return name.toLowerCase().endsWith(".png");
            }
        });

        graphicsCount = files.length;
        if (graphicsCount > 0) {
            graphicsImages = new PImage[graphicsCount];

            for (int i = 0; i < graphicsCount; i++) {
                PImage img = loadImage(files[i].getAbsolutePath());
                graphicsImages[i] = img;
            }
        } else {
            println("Keine PNG-Dateien im CustomGraphics-Ordner gefunden.");
        }
    } else {
        println("CustomGraphics-Ordner nicht gefunden oder ist kein Verzeichnis.");
    }
}

void saveTransparentPng() {
    PGraphics pg = createGraphics(canvasWidth, canvasHeight);
    pg.beginDraw();
    pg.clear();
    pg.background(255, 255, 255, 0);
    pg.imageMode(CORNER);
    pg.textFont(unicodeFont);

    if (showGrid) {
        pg.strokeWeight(1);
        int beatsPerBar = 4;
        float barWidth = canvasWidth / (float) bars;
        float beatWidth = barWidth / beatsPerBar;

        for (int i = 0; i <= bars * beatsPerBar; i++) {
            float x = i * beatWidth;
            if (i % beatsPerBar == 0) {
                pg.strokeWeight(3);
            } else {
                pg.strokeWeight(1);
            }
            pg.line(x, 0, x, canvasHeight);
        }
    }

    if (showNotes) {
        if (circleMode) {
            for (Track track : sequence.getTracks()) {
                for (int i = 0; i < track.size(); i++) {
                    MidiEvent event = track.get(i);
                    MidiMessage message = event.getMessage();
                    if (message instanceof ShortMessage) {
                        ShortMessage sm = (ShortMessage) message;
                        if (sm.getCommand() == ShortMessage.NOTE_ON) {
                            int key = sm.getData1();
                            int velocity = sm.getData2();
                            if (velocity > 0) {
                                long tick = event.getTick();
                                float x = map(tick, 0, bars * 4 * ppq, 0, canvasWidth);
                                float y = map(key, 0, 127, canvasHeight, 0) * ySpacing + getNoteYOffset(tick, key);
                                float adjustedSize = multiCircle * size * (1 + (velocity * velocityInfluence / 127.0));
                                float rotation = getNoteRotation(tick, key) * rotationStrength;
                                pg.pushMatrix();
                                pg.translate(x + offsetX, y + offsetY);
                                pg.rotate(rotation);
                                pg.fill(0);
                                pg.noStroke();
                                pg.ellipse(0, 0, adjustedSize, adjustedSize);
                                pg.popMatrix();
                            }
                        }
                    }
                }
            }
        } else if (unicodeMode) {
            for (Track track : sequence.getTracks()) {
                for (int i = 0; i < track.size(); i++) {
                    MidiEvent event = track.get(i);
                    MidiMessage message = event.getMessage();
                    if (message instanceof ShortMessage) {
                        ShortMessage sm = (ShortMessage) message;
                        if (sm.getCommand() == ShortMessage.NOTE_ON) {
                            int key = sm.getData1();
                            int velocity = sm.getData2();
                            if (velocity > 0) {
                                long tick = event.getTick();
                                float x = map(tick, 0, bars * 4 * ppq, 0, canvasWidth);
                                float y = map(key, 0, 127, canvasHeight, 0) * ySpacing + getNoteYOffset(tick, key);
                                float adjustedSize = multiUnicode * size * (1 + (velocity * velocityInfluence / 127.0));
                                float rotation = getNoteRotation(tick, key) * rotationStrength;
                                pg.pushMatrix();
                                pg.translate(x + offsetX, y + offsetY);
                                pg.translate(0, -adjustedSize / 2); // Verschiebt den Ursprungspunkt der Translation
                                pg.rotate(radians(rotationAngle)); // Rotiert die Grafik um den angegebenen Winkel
                                pg.rotate(rotation);
                                pg.fill(0);
                                pg.textAlign(LEFT, CENTER);
                                pg.textSize(adjustedSize);
                                int symbolIndex = (key + variance) % unicodeSymbols.length;
                                pg.text(unicodeSymbols[symbolIndex], 0, 0);
                                pg.popMatrix();
                            }
                        }
                    }
                }
            }
        } else if (graphicsMode) {
            for (Track track : sequence.getTracks()) {
                for (int i = 0; i < track.size(); i++) {
                    MidiEvent event = track.get(i);
                    MidiMessage message = event.getMessage();
                    if (message instanceof ShortMessage) {
                        ShortMessage sm = (ShortMessage) message;
                        if (sm.getCommand() == ShortMessage.NOTE_ON) {
                            int key = sm.getData1();
                            int velocity = sm.getData2();
                            if (velocity > 0) {
                                long tick = event.getTick();
                                float x = map(tick, 0, bars * 4 * ppq, 0, canvasWidth);
                                float y = map(key, 0, 127, canvasHeight, 0) * ySpacing + getNoteYOffset(tick, key);
                                float scaleFactor = 1 + (velocity * velocityInfluence / 127.0);
                                float adjustedWidth = multiGraphic * size * scaleFactor;
                                float adjustedHeight = adjustedWidth * ((float) graphicsImages[0].height / graphicsImages[0].width);
                                float rotation = getNoteRotation(tick, key) * rotationStrength;

                                float translateX = x + offsetX;
                                float translateY = y + offsetY;

                                pg.pushMatrix();
                                pg.translate(translateX, translateY);
                                pg.translate(0, -adjustedHeight / 2); // Verschiebt den Ursprungspunkt der Translation
                                pg.rotate(radians(rotationAngle)); // Rotiert die Grafik um den angegebenen Winkel
                                pg.rotate(rotation);
                                pg.imageMode(CORNER);
                                pg.image(graphicsImages[(key + variance) % graphicsCount], 0, 0, adjustedWidth, adjustedHeight);
                                pg.popMatrix();
                            }
                        }
                    }
                }
            }
        }
    }

    if (showName) {
        pg.fill(0);
        pg.textFont(buttonFont);
        pg.textAlign(LEFT, TOP);
        String fileName = new File(midiFilePath).getName();
        pg.text(fileName, 10, 10);
    }

    pg.endDraw();

    String timestamp = new SimpleDateFormat("HHmm").format(new Date());
    String fileName = String.format(exportFolderPath + "/%s_S%d_OX%d_OY%d_YS%.1f_V%d_R%.2f_YR%.2f_%s.png",
                                    midiFilePath.substring(midiFilePath.lastIndexOf("\\") + 1, midiFilePath.lastIndexOf(".")),
                                    size, offsetX, offsetY, ySpacing, variance, rotationStrength, randomizeY, timestamp);
    pg.save(fileName);
    println("Gespeicherte Datei: " + fileName);

    notificationVisible = true;
    notificationStartTime = millis();
}

int getBarsFromFileName(String filePath, int inputBars) {
    String fileName = new File(filePath).getName();
    String barsString = "";

    int lastUnderscoreIndex = fileName.lastIndexOf('_');
    int barIndex = fileName.lastIndexOf("Bar");

    if (lastUnderscoreIndex != -1 && barIndex != -1 && barIndex > lastUnderscoreIndex) {
        barsString = fileName.substring(lastUnderscoreIndex + 1, barIndex);
    }

    int bars = inputBars; // Verwende die Eingabevariable inputBars als Standardwert
    try {
        if (!barsString.isEmpty()) {
            bars = Integer.parseInt(barsString);
        } else {
            bars = inputBars; // Verwende inputBars, wenn barsString leer ist
        }
    } catch (NumberFormatException e) {
        println("Fehler beim Extrahieren der Takte aus dem Dateinamen. Standardwert wird verwendet: " + bars);
    }

    return bars;
}


class Slider {

    float x, y; // Position des Sliders
    float width, height; // Abmessungen des Sliders
    float minValue, maxValue; // Minimaler und maximaler Wert des Sliders
    float value; // Aktueller Wert des Sliders
    boolean dragging = false; // Gibt an, ob der Slider gezogen wird
    float buttonRadius = 6; // Radius des Slider-Knopfes

    Slider(float x, float y, float width, float height, float minValue, float maxValue, float startValue) {
        this.x = x;
        this.y = y;
        this.width = width;
        this.height = height;
        this.minValue = minValue;
        this.maxValue = maxValue;
        this.value = startValue;
    }

    void display() {
        // Zeichne den Pfeil
        stroke(0);
        fill(0);
        float buttonX = map(value, minValue, maxValue, x + buttonRadius, x + width - buttonRadius);
        float buttonY = y + height / 2;
        
          // Zeichne den weißen Rand der Linie
  stroke(255); // Weißer Rand
  strokeWeight(2+arrowStroke); // Dickere Linie für den Rand
  line(x, buttonY, buttonX, buttonY); // Zeichne den Linienanteil des Pfeils
  
  // Zeichne die schwarze Linie
  stroke(0); // Schwarze Linie
  strokeWeight(2); // Dünnere Linie
  line(x, buttonY, buttonX, buttonY); // Zeichne den Linienanteil des Pfeils


  // Dynamische Größe des Pfeilkopfs basierend auf der X-Position
          float arrowSize = map(buttonX, x + buttonRadius, x + width - buttonRadius, buttonRadius, dynamicButtonRadiusMax);

        pushMatrix();
        translate(buttonX, buttonY);
        stroke(255); // Weißer Stroke
        strokeWeight(arrowStroke); // Strichstärke von 0.5
        fill(0); // Schwarze Füllung
        beginShape();
        vertex(-arrowSize, arrowSize);        
        vertex(-arrowSize, -arrowSize);
        vertex(arrowSize, 0);
        endShape(CLOSE);
        popMatrix();
    }

  float hoverRadius = 50; // Fester Radius für die Ellipse

void mousePressed() {
  // Prüfe, ob die Maus über dem Knopf ist
  float buttonX = map(value, minValue, maxValue, x + buttonRadius, x + width - buttonRadius);
  if (mouseX > buttonX - hoverRadius && mouseX < buttonX + hoverRadius && mouseY > y && mouseY < y + height) {
    dragging = true;
  }
}
    void mouseReleased() {
        dragging = false;
    }

    void mouseDragged() {
        if (dragging) {
            // Begrenze die Position des Knopfes innerhalb des Sliders
            float buttonX = constrain(mouseX, x + buttonRadius, x + width - buttonRadius);
            // Übersetze die Position des Knopfes in den Wert des Sliders
            value = map(buttonX, x + buttonRadius, x + width - buttonRadius, minValue, maxValue);
            // Passe die Größe, Verschiebung, Varianz und Rotationsstärke entsprechend dem Slider-Wert an
            if (this == sizeSlider) {
                size = (int) value; // Wert in int umwandeln für die Größe
            } else if (this == velocityInfluenceSlider) {
                velocityInfluence = value; // velocityInfluence bleibt ein float
            } else if (this == varianceSlider) {
                variance = (int) value; // Wert in int umwandeln für die Varianz
            } else if (this == offsetXSlider) {
                offsetX = (int) value; // Wert in int umwandeln für die X-Verschiebung
            } else if (this == offsetYSlider) {
                offsetY = (int) value; // Wert in int umwandeln für die Y-Verschiebung
            } else if (this == ySpacingSlider) {
                ySpacing = value; // ySpacing bleibt ein float
            } else if (this == rotationStrengthSlider) {
                rotationStrength = value; // rotationStrength bleibt ein float
            } else if (this == randomizeYSlider) {
                randomizeY = value; // randomizeY bleibt ein float
            } else if (this == rotationSlider) {
                rotationAngle = (int) value; // Wert in int umwandeln für die Rotation
            }
        }
    }
}

void updateNoteRotations() {
    noteRotations.clear(); // Bestehende Rotationen löschen
    for (Track track : sequence.getTracks()) {
        for (int i = 0; i < track.size(); i++) {
            MidiEvent event = track.get(i);
            MidiMessage message = event.getMessage();
            if (message instanceof ShortMessage) {
                ShortMessage sm = (ShortMessage) message;
                if (sm.getCommand() == ShortMessage.NOTE_ON) {
                    int key = sm.getData1();
                    long tick = event.getTick();
                    String noteKey = tick + "_" + key; // Kombiniere Tick und Key, um einen eindeutigen Schlüssel zu erstellen
                    noteRotations.put(noteKey, random(-1, 1)); // Rotationen zwischen -1 und 1 zuweisen
                }
            }
        }
    }
}

void updateNoteYOffsets() {
    noteYOffsets.clear(); // Bestehende Y-Offsets löschen
    for (Track track : sequence.getTracks()) {
        for (int i = 0; i < track.size(); i++) {
            MidiEvent event = track.get(i);
            MidiMessage message = event.getMessage();
            if (message instanceof ShortMessage) {
                ShortMessage sm = (ShortMessage) message;
                if (sm.getCommand() == ShortMessage.NOTE_ON) {
                    int key = sm.getData1();
                    long tick = event.getTick();
                    String noteKey = tick + "_" + key; // Kombiniere Tick und Key, um einen eindeutigen Schlüssel zu erstellen
                    noteYOffsets.put(noteKey, random(-1, 1)); // Y-Offsets zwischen -1 und 1 zuweisen
                }
            }
        }
    }
}

float getNoteRotation(long tick, int key) {
    String noteKey = tick + "_" + key;
    if (!noteRotations.containsKey(noteKey)) {
        noteRotations.put(noteKey, random(-1, 1)); // Rotationen zwischen -1 und 1 zuweisen, wenn noch nicht vorhanden
    }
    return noteRotations.get(noteKey);
}

float getNoteYOffset(long tick, int key) {
    String noteKey = tick + "_" + key;
    if (!noteYOffsets.containsKey(noteKey)) {
        noteYOffsets.put(noteKey, random(-1, 1)); // Y-Offsets zwischen -1 und 1 zuweisen, wenn noch nicht vorhanden
    }
    return noteYOffsets.get(noteKey) * randomizeY; // Multipliziere den gespeicherten Offset mit der aktuellen Stärke
}

void randomizeNoteAttributes() {
    updateNoteRotations();
    updateNoteYOffsets();
}

void mousePressed() {
    // Prüfe, ob der Save-Button angeklickt wurde
    if (saveButtonHovered) {
        saveTransparentPng();
    }
    // Prüfe, ob der Toggle Grid-Button angeklickt wurde
    if (toggleGridButtonHovered) {
        showGrid = !showGrid;
    }
    // Prüfe, ob der Toggle Note-Button angeklickt wurde
    if (toggleNoteButtonHovered) {
        if (circleMode) {
            circleMode = false;
        } else {
            circleMode = true;
            unicodeMode = false; // Schalte Unicode-Modus aus, wenn Kreis-Modus aktiviert wird
            graphicsMode = false; // Schalte Grafik-Modus aus, wenn Kreis-Modus aktiviert wird
        }
    }
    // Prüfe, ob der Toggle Unicode-Button angeklickt wurde
    if (toggleUnicodeButtonHovered) {
        if (unicodeMode) {
            unicodeMode = false;
        } else {
            unicodeMode = true;
            circleMode = false; // Schalte Kreis-Modus aus, wenn Unicode-Modus aktiviert wird
            graphicsMode = false; // Schalte Grafik-Modus aus, wenn Unicode-Modus aktiviert wird
        }
    }
    // Prüfe, ob der Toggle Graphics-Button angeklickt wurde
    if (toggleGraphicsButtonHovered) {
        if (graphicsMode) {
            graphicsMode = false;
        } else {
            graphicsMode = true;
            circleMode = false; // Schalte Kreis-Modus aus, wenn Grafik-Modus aktiviert wird
            unicodeMode = false; // Schalte Unicode-Modus aus, wenn Grafik-Modus aktiviert wird
        }
    }
    // Prüfe, ob der Toggle Name-Button angeklickt wurde
    if (toggleNameButtonHovered) {
        showName = !showName;
    }
    // Prüfe, ob der Toggle UI-Button angeklickt wurde
    if (toggleUIButtonHovered) {
        showUI = !showUI;
    }
    // Prüfe, ob der Reset Values-Button angeklickt wurde
    if (resetValuesButtonHovered) {
        resetValues();
    }
    // Prüfe, ob der Randomize-Button angeklickt wurde
    if (randomizeButtonHovered) {
        randomizeNoteAttributes();
    }

    // Prüfe, ob die Maus auf den Slidern gedrückt wurde
    sizeSlider.mousePressed();
    velocityInfluenceSlider.mousePressed();
    varianceSlider.mousePressed();
    offsetXSlider.mousePressed();
    offsetYSlider.mousePressed();
    ySpacingSlider.mousePressed();
    rotationStrengthSlider.mousePressed();
    randomizeYSlider.mousePressed(); // Hinzugefügt: Y-Randomisierungs-Slider,
    rotationSlider.mousePressed(); // Hinzugefügt: Rotation-Slider

}

void mouseReleased() {
    // Slider freigeben, wenn die Maus losgelassen wird
    sizeSlider.mouseReleased();
    velocityInfluenceSlider.mouseReleased();
    varianceSlider.mouseReleased();
    offsetXSlider.mouseReleased();
    offsetYSlider.mouseReleased();
    ySpacingSlider.mouseReleased();
    rotationStrengthSlider.mouseReleased();
    randomizeYSlider.mouseReleased(); // Hinzugefügt: Y-Randomisierungs-Slider,
    rotationSlider.mouseReleased(); // Hinzugefügt: Rotation-Slider

}

void mouseDragged() {
    // Slider ziehen, wenn die Maus gezogen wird
    sizeSlider.mouseDragged();
    velocityInfluenceSlider.mouseDragged();
    varianceSlider.mouseDragged();
    offsetXSlider.mouseDragged();
    offsetYSlider.mouseDragged();
    ySpacingSlider.mouseDragged();
    rotationStrengthSlider.mouseDragged();
    randomizeYSlider.mouseDragged(); // Hinzugefügt: Y-Randomisierungs-Slider
    rotationSlider.mouseDragged(); // Hinzugefügt: Rotation-Slider

}

void saveSettings() {
    JSONObject settings = new JSONObject();
    settings.setInt("size", size);
    settings.setInt("offsetX", offsetX);
    settings.setInt("offsetY", offsetY);
    settings.setFloat("ySpacing", ySpacing);
    settings.setInt("variance", variance);
    settings.setFloat("velocityInfluence", velocityInfluence);
    settings.setFloat("rotationStrength", rotationStrength);
    settings.setFloat("randomizeY", randomizeY); // Hinzugefügt: Y-Randomisierung speichern

    saveJSONObject(settings, settingsFilePath);
}

void loadSettings() {
    File settingsFile = new File(settingsFilePath);
    if (settingsFile.exists()) {
        try {
            JSONObject settings = loadJSONObject(settingsFilePath);
            size = settings.getInt("size", defaultSize);
            offsetX = settings.getInt("offsetX", defaultOffsetX);
            offsetY = settings.getInt("offsetY", defaultOffsetY);
            ySpacing = settings.getFloat("ySpacing", defaultYSpacing);
            variance = settings.getInt("variance", defaultVariance);
            velocityInfluence = settings.getFloat("velocityInfluence", defaultVelocityInfluence);
            rotationStrength = settings.getFloat("rotationStrength", defaultRotationStrength);
            randomizeY = settings.getFloat("randomizeY", defaultRandomizeY);
            rotationAngle = settings.getInt("rotationAngle", 0); // Hinzugefügt: Rotation laden
        } catch (Exception e) {
            println("Fehler beim Laden der Einstellungen: " + e.getMessage());
        }
    }
}

void resetValues() {
    size = defaultSize;
    offsetX = defaultOffsetX;
    offsetY = defaultOffsetY;
    ySpacing = defaultYSpacing;
    variance = defaultVariance;
    velocityInfluence = defaultVelocityInfluence;
    rotationStrength = defaultRotationStrength;
    randomizeY = defaultRandomizeY;
    rotationAngle = 0; // Hinzugefügt: Rotation zurücksetzen

    // Sliderwerte ebenfalls zurücksetzen
    sizeSlider.value = size;
    velocityInfluenceSlider.value = velocityInfluence;
    varianceSlider.value = variance;
    offsetXSlider.value = offsetX;
    offsetYSlider.value = offsetY;
    ySpacingSlider.value = ySpacing;
    rotationStrengthSlider.value = rotationStrength;
    randomizeYSlider.value = randomizeY;
    rotationSlider.value = rotationAngle; // Hinzugefügt: Rotation-Slider
}

void exit() {
    saveSettings();
    super.exit();
}

void drawNotification() {
    fill(0, 102, 204);
    rect(width - 210, 10, 200, 30, 10);
    fill(255);
    textAlign(CENTER, CENTER);
    text("PNG erfolgreich gespeichert!", width - 110, 25);

    // Überprüfen, ob die Benachrichtigung für 2 Sekunden angezeigt wurde
    if (millis() - notificationStartTime > 2000) {
        notificationVisible = false;
    }
}
